<?php
// core/Bot.php
require_once 'Database.php';
require_once 'FileManager.php';

class Bot {
    private $token;
    private $db;
    private $fileManager;
    private $apiUrl = "https://api.telegram.org/bot";
    
    public function __construct($token) {
        $this->token = $token;
        $this->db = Database::getInstance();
        $this->fileManager = new FileManager();
    }
    
    public function start() {
        $update = json_decode(file_get_contents('php://input'), true);
        
        if (isset($update['message'])) {
            $this->handleMessage($update['message']);
        } elseif (isset($update['callback_query'])) {
            $this->handleCallback($update['callback_query']);
        }
    }
    
    private function handleMessage($message) {
        $chatId = $message['chat']['id'];
        $text = $message['text'] ?? '';
        $file = $message['document'] ?? $message['photo'] ?? null;
        
        if ($file) {
            $this->handleFileUpload($chatId, $file);
        } elseif (strpos($text, '/start') === 0) {
            $this->sendWelcome($chatId);
        } elseif (strpos($text, '/download') === 0) {
            $this->handleDownloadRequest($chatId, $text);
        } else {
            $this->sendMessage($chatId, "دستور نامعتبر! از /start استفاده کنید.");
        }
    }
    
    private function handleFileUpload($chatId, $file) {
        // اگر فایل عکس بود (آرایه‌ایه)، آخرین کیفیت رو بگیر
        if (is_array($file) && isset($file['file_id'])) {
            $fileId = $file['file_id'];
            $fileType = 'photo';
        } elseif (isset($file['file_id'])) {
            $fileId = $file['file_id'];
            $fileType = 'document';
        } else {
            $this->sendMessage($chatId, "خطا در دریافت فایل!");
            return;
        }
        
        // دریافت اطلاعات فایل از تلگرام
        $fileInfo = $this->apiRequest('getFile', ['file_id' => $fileId]);
        if (!$fileInfo || !$fileInfo['ok']) {
            $this->sendMessage($chatId, "خطا در پردازش فایل!");
            return;
        }
        
        $filePath = $fileInfo['result']['file_path'];
        $telegramFileUrl = "https://api.telegram.org/file/bot{$this->token}/{$filePath}";
        
        // ذخیره فایل در سرور
        $savedFileId = $this->fileManager->saveFileFromUrl($telegramFileUrl);
        if ($savedFileId) {
            $downloadLink = $this->fileManager->getDownloadLink($savedFileId);
            $this->sendMessage(
                $chatId,
                "✅ فایل با موفقیت آپلود شد!\n\n" .
                "🔗 لینک دانلود:\n" . $downloadLink . "\n\n" .
                "این لینک رو با بقیه به اشتراک بذار!"
            );
        } else {
            $this->sendMessage($chatId, "❌ خطا در ذخیره‌سازی فایل!");
        }
    }
    
    private function sendWelcome($chatId) {
        $text = "👋 به *آپلودر پرو* خوش آمدید!\n\n";
        $text .= "فایل (عکس، ویدیو، سند، ...) رو اینجا بفرست تا برات لینک دانلود دائمی بسازم!\n\n";
        $text .= "توسط این لینک هرکسی می‌تونه فایل رو از طریق ربات دریافت کنه، حتی اگر عضو ربات نباشه!";
        
        $this->sendMessage($chatId, $text);
    }
    
    private function handleDownloadRequest($chatId, $text) {
        $parts = explode(' ', $text);
        if (count($parts) < 2) {
            $this->sendMessage($chatId, "⚠️ لطفاً لینک دانلود رو وارد کنید!");
            return;
        }
        
        $fileId = $this->fileManager->extractFileIdFromUrl($parts[1]);
        if (!$fileId) {
            $this->sendMessage($chatId, "❌ لینک نامعتبر!");
            return;
        }
        
        $filePath = $this->fileManager->getFilePath($fileId);
        if ($filePath && file_exists($filePath)) {
            // ارسال فایل به کاربر
            $this->sendDocument($chatId, $filePath);
        } else {
            $this->sendMessage($chatId, "❌ فایل مورد نظر یافت نشد!");
        }
    }
    
    private function sendMessage($chatId, $text, $parseMode = 'Markdown') {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => $parseMode
        ];
        $this->apiRequest('sendMessage', $params);
    }
    
    private function sendDocument($chatId, $filePath) {
        $params = [
            'chat_id' => $chatId,
            'document' => new CURLFile(realpath($filePath))
        ];
        $this->apiRequest('sendDocument', $params);
    }
    
    private function apiRequest($method, $params = []) {
        $url = $this->apiUrl . $this->token . '/' . $method;
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $result = curl_exec($ch);
        curl_close($ch);
        return json_decode($result, true);
    }
}
?>